<?php

namespace App\Services;

use App\Models\User;
use App\Models\Market;
use App\Models\UserBalance;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class WalletService
{
    /**
     * Ensure user has a balance record for each market
     * Creates missing balance records initialized to 0
     * This is a reusable function to call anywhere in your app
     */
    public static function ensureUserHasAllMarketBalances(User $user): void
    {
        // Get all active markets
        $markets = Market::all();

        // Get existing balances for this user
        $existingBalances = UserBalance::where('user_id', $user->id)
            ->pluck('market_id')
            ->toArray();

        // Create missing balance records
        $missingMarkets = $markets->whereNotIn('id', $existingBalances);

        if ($missingMarkets->isNotEmpty()) {
            $balancesToInsert = $missingMarkets->map(function ($market) use ($user) {
                return [
                    'user_id' => $user->id,
                    'market_id' => $market->id,
                    'balance' => 0,
                    'uuid' => Str::uuid(),
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            })->toArray();

            UserBalance::insert($balancesToInsert);
        }
    }

    /**
     * Get user's balance for a specific market
     */
    public static function getUserMarketBalance(User $user, Market $market): ?UserBalance
    {
        return UserBalance::where('user_id', $user->id)
            ->where('market_id', $market->id)
            ->first();
    }

    /**
     * Get all user balances
     */
    public static function getUserBalances(User $user)
    {
        return UserBalance::where('user_id', $user->id)
            ->with('market')
            ->get();
    }

    /**
     * Check if user has sufficient balance in a market
     */
    public static function hasSufficientBalance(User $user, Market $market, float $amount): bool
    {
        $balance = self::getUserMarketBalance($user, $market);
        return $balance && $balance->hasSufficientBalance($amount);
    }

    /**
     * Get total balance across all markets
     */
    public static function getTotalBalance(User $user): float
    {
        return (float) UserBalance::where('user_id', $user->id)->sum('balance');
    }

    /**
     * Deduct balance from a market
     */
    public static function deductBalance(User $user, Market $market, float $amount): void
    {
        $balance = self::getUserMarketBalance($user, $market);
        if (!$balance) {
            throw new \Exception("User does not have balance for market: {$market->asset}");
        }

        if (!$balance->hasSufficientBalance($amount)) {
            throw new \Exception("Insufficient balance in {$market->asset}. Required: {$amount}, Available: {$balance->balance}");
        }

        $balance->deductBalance($amount);
    }

    /**
     * Add balance to a market
     */
    public static function addBalance(User $user, Market $market, float $amount): void
    {
        $balance = self::getUserMarketBalance($user, $market);
        if (!$balance) {
            throw new \Exception("User does not have balance for market: {$market->asset}");
        }

        $balance->addBalance($amount);
    }
}
