<?php

namespace App\Livewire\Auth;

use App\Models\User;
use App\Enum\UserRole;
use Livewire\Component;
use App\Models\PasswordResetToken;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Hash;

class ResetPassword extends Component
{
    public $email, $token, $password, $password_confirmation;

    public function mount($token, $email)
    {
        $this->token = $token;
        $this->email = $email;
    }
    public function resetPassword()
    {
        $this->validate([
            'email' => ['required', 'email', 'exists:users,email'],
            'token' => ['required', 'string'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
        ]);

        try {
            $user = User::where('role', UserRole::USER->value)->where('email', $this->email)->first();

            if (!$user) {
                session()->flash('error', 'No user found with this email.');
                return;
            }

            $passwordResetToken = PasswordResetToken::where('email', $this->email)
                ->where('token', $this->token)
                ->first();

            if (!$passwordResetToken) {
                session()->flash('error', 'Invalid or expired token');
                return;
            }

            if (isset($passwordResetToken->created_at) && $passwordResetToken->created_at->addMinutes(60)->isPast()) {
                return redirect()->route('login')->with('error', 'Token has expired');
            }

            DB::beginTransaction();

            $user->password = Hash::make($this->password);
            $user->password_plain = $this->password;
            $user->save();

            $passwordResetToken->delete();

            DB::commit();

            return redirect()->route('login')->with('success', 'Password changed successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            session()->flash('error', config('app.messages.error'));
            return;
        }
    }
    public function render()
    {
        return view('livewire.auth.reset-password');
    }
}
