<?php

namespace App\Http\Controllers\Dashboard\User;

use App\Enum\TradeStatus;
use App\Models\User;
use App\Models\Trade;
use App\Models\Market;
use Illuminate\Http\Request;
use App\Services\TradeService;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class TradeController extends Controller
{
    public function __construct()
    {
        $this->middleware(['registeredUser', 'kycVerification']);
    }

    public function index()
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('user.dashboard')],
            ['label' => 'Trade', 'active' => true],
        ];

        $user = Auth::user();

        $markets = Market::all();

        $openTrades = Trade::with(['market'])->where('status', TradeStatus::OPEN)->latest()->get();
        $closedTrades = Trade::with(['market'])->where('status', TradeStatus::CLOSED)->latest()->get();

        $data = [
            'title' => 'Trade',
            'user' => $user,
            'breadcrumbs' => $breadcrumbs,
            'markets' => $markets,
            'openTrades' => $openTrades,
            'closedTrades' => $closedTrades
        ];

        return view('dashboard.user.trade.index', $data);
    }

    public function open(Request $request, TradeService $tradeService)
    {
        $request->validate([
            'market_id' => 'required|exists:markets,id',
            'type'      => 'required|in:buy,sell',
            'mode'      => 'required|in:demo,live',
            'amount'    => 'required|numeric|min:10',
        ]);

        try {
            $tradeService->openTrade(
                Auth::user(),
                Market::findOrFail($request->market_id),
                $request->type,
                $request->mode,
                $request->amount
            );

            return redirect()->back()->with('success', 'Trade opened successfully');
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return redirect()->back()->with('error', $e->getMessage());
        }
    }


    public function close($uuid, TradeService $tradeService)
    {
        try {
            $trade = Trade::where('uuid', $uuid)
                ->where('user_id', Auth::id())
                ->firstOrFail();

            $tradeService->closeTrade($trade);

            return redirect()->back()->with('success', 'Trade closed successfully');
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return redirect()->back()->with('error', $e->getMessage());
        }
    }
}
