<?php

namespace App\Http\Controllers\Dashboard\User;

use App\Models\Plan;
use App\Models\User;
use App\Models\Wallet;
use App\Models\UserPlan;
use App\Trait\FileUpload;
use App\Enum\WalletStatus;
use App\Models\Transaction;
use App\Enum\UserPlanStatus;
use Illuminate\Http\Request;
use App\Enum\TransactionType;
use App\Enum\TransactionMethod;
use App\Enum\TransactionStatus;
use App\Enum\TransactionDirection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;

class PlanController extends Controller
{

    use FileUpload;

    public function __construct()
    {
        $this->middleware(['registeredUser', 'kycVerification']);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $user = Auth::user();

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('user.dashboard')],
            ['label' => 'Trading Plans', 'active' => true]
        ];

        $plans = Plan::latest()->get();

        $data = [
            'title' => 'Trading Plans',
            'user' => $user,
            'breadcrumbs' => $breadcrumbs,
            'plans' => $plans
        ];

        return view('dashboard.user.plan.index', $data);
    }

    public function store(Request $request, string $uuid)
    {
        $plan = Plan::where('uuid', $uuid)->firstOrFail();

        // Build dynamic field name
        $field = 'amount_' . $uuid;

        $request->validate([
            $field => ['required', 'numeric', 'min:' . $plan->minimum, 'max:' . $plan->maximum],
        ], [
            $field . '.min' => 'Amount is below the minimum required for this plan.',
            $field . '.max' => 'Amount exceeds the maximum allowed for this plan.',
            $field . '.required' => 'Amount is required.',
            $field . '.numeric' => 'Amount must be a number.',
        ]);

        // Access the validated amount
        $amount = $request->input($field);

        try {
            DB::beginTransaction();

            // Lock user row
            $user = User::where('id', Auth::id())
                ->lockForUpdate()
                ->first();

            if ($user->balance < $amount) {
                throw new \Exception('Insufficient balance.');
            }

            $startsAt = now();
            $endsAt = (clone $startsAt)->addDays($plan->daily_profit_duration);

            $userPlan = UserPlan::create([
                'user_id' => $user->id,
                'plan_id' => $plan->id,
                'amount' => $amount,
                'daily_profit' => $plan->daily_profit,
                'duration' => $plan->daily_profit_duration,
                'trades_per_day' => $plan->trades_per_day,
                'starts_at' => $startsAt,
                'ends_at' => $endsAt,
                'status' => UserPlanStatus::ACTIVE
            ]);

            // Deduct balance
            $user->balance -= $amount;
            $user->save();

            $meta = [
                'plan' => [
                    'name' => $plan->name,
                    'type' => $plan->type,
                    'daily_profit' => $plan->daily_profit,
                    'duration' => $plan->daily_profit_duration,
                    'trades_per_day' => $plan->trades_per_day,
                ]
            ];

            Transaction::create([
                'user_id' => $user->id,
                'type' => TransactionType::PLAN_SUBSCRIPTION,
                'direction' => TransactionDirection::DEBIT,
                'description' => 'Plan Subscription - ' . $plan->name,
                'amount' => $amount,
                'method' => TransactionMethod::CRYPTO,
                'transaction_at' => now(),
                'reference_id' => generateReferenceId(),
                'other_reference_id' => $userPlan->id,
                'meta' => $meta,
                'status' => TransactionStatus::COMPLETED
            ]);

            DB::commit();

            return redirect()->route('user.plan.index')
                ->with('success', 'Plan subscribed successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());

            return redirect()->route('user.plan.index')
                ->with('error', 'Plan subscription failed. Please try again.');
        }
    }
}
