<?php

namespace App\Http\Controllers\Dashboard\User;

use App\Models\User;
use App\Models\Wallet;
use App\Trait\FileUpload;
use App\Enum\WalletStatus;
use App\Models\IsolatedMarginRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;

class IsolatedMarginRequestController extends Controller
{
    use FileUpload;

    public function __construct()
    {
        $this->middleware(['registeredUser', 'kycVerification']);
    }

    /**
     * Display isolated margin information and history
     */
    public function index()
    {
        $user = Auth::user();

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('user.dashboard')],
            ['label' => 'Isolated Margin', 'active' => true]
        ];

        $requests = IsolatedMarginRequest::where('user_id', $user->id)->latest()->get();

        $data = [
            'title' => 'Isolated Margin',
            'user' => $user,
            'breadcrumbs' => $breadcrumbs,
            'requests' => $requests
        ];

        return view('dashboard.user.isolated_margin.index', $data);
    }

    /**
     * Select wallet method for isolated margin upgrade
     */
    public function method()
    {
        $user = Auth::user();

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('user.dashboard')],
            ['label' => 'Isolated Margin', 'url' => route('user.isolated_margin.index')],
            ['label' => 'Select Method', 'active' => true],
        ];

        $wallets = Wallet::where('status', WalletStatus::ACTIVE)->get();

        $data = [
            'title' => 'Select Payment Method',
            'user' => $user,
            'breadcrumbs' => $breadcrumbs,
            'wallets' => $wallets
        ];

        return view('dashboard.user.isolated_margin.method', $data);
    }

    /**
     * Show payment details for isolated margin upgrade
     */
    public function payment(string $uuid)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('user.dashboard')],
            ['label' => 'Isolated Margin', 'url' => route('user.isolated_margin.index')],
            ['label' => 'Select Method', 'url' => route('user.isolated_margin.method')],
            ['label' => 'Payment Details', 'active' => true],
        ];

        $wallet = Wallet::where('uuid', $uuid)->firstOrFail();

        $data = [
            'title' => 'Payment Details',
            'breadcrumbs' => $breadcrumbs,
            'wallet' => $wallet
        ];

        return view('dashboard.user.isolated_margin.payment', $data);
    }

    /**
     * Show payment proof upload page
     */
    public function paymentProof(string $uuid)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('user.dashboard')],
            ['label' => 'Isolated Margin', 'url' => route('user.isolated_margin.index')],
            ['label' => 'Select Method', 'url' => route('user.isolated_margin.method')],
            ['label' => 'Payment Details', 'url' => route('user.isolated_margin.payment', $uuid)],
            ['label' => 'Upload Proof', 'active' => true],
        ];

        $wallet = Wallet::where('uuid', $uuid)->firstOrFail();

        $data = [
            'title' => 'Upload Payment Proof',
            'breadcrumbs' => $breadcrumbs,
            'wallet' => $wallet
        ];

        return view('dashboard.user.isolated_margin.payment_proof', $data);
    }

    /**
     * Store isolated margin request
     */
    public function store(Request $request, string $uuid)
    {
        $request->validate([
            'amount' => ['required', 'numeric', 'min:1'],
            'transaction_hash' => ['required', 'string'],
            'proof' => ['required', 'image', 'mimes:jpeg,png,jpg,gif,svg', 'max:2048'],
        ]);

        $user = Auth::user();
        $wallet = Wallet::where('uuid', $uuid)->firstOrFail();

        try {
            DB::beginTransaction();

            IsolatedMarginRequest::create([
                'user_id' => $user->id,
                'wallet_id' => $wallet->id,
                'amount' => $request->amount,
                'proof' => $this->updateFile($request, 'proof', uploadPathUser('isolated_margin'), null),
            ]);

            DB::commit();

            return redirect()->route('user.isolated_margin.index')
                ->with('success', 'Your isolated margin upgrade request has been submitted. Admin will review and approve it shortly.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->route('user.isolated_margin.index')
                ->with('error', config('app.messages.error'));
        }
    }

    /**
     * Show isolated margin request details
     */
    public function show(string $uuid)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('user.dashboard')],
            ['label' => 'Isolated Margin', 'url' => route('user.isolated_margin.index')],
            ['label' => 'Request Details', 'active' => true],
        ];

        $user = Auth::user();
        $request = IsolatedMarginRequest::where('uuid', $uuid)
            ->where('user_id', $user->id)
            ->firstOrFail();

        $data = [
            'title' => 'Isolated Margin Request Details',
            'breadcrumbs' => $breadcrumbs,
            'user' => $user,
            'request' => $request
        ];

        return view('dashboard.user.isolated_margin.show', $data);
    }
}
