<?php

namespace App\Http\Controllers\Dashboard\User;

use App\Models\CopyTradePlan;
use App\Models\User;
use App\Models\UserCopyTradePlan;
use App\Models\Transaction;
use App\Enum\UserCopyTradePlanStatus;
use App\Enum\TransactionType;
use App\Enum\TransactionMethod;
use App\Enum\TransactionStatus;
use App\Enum\TransactionDirection;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;

class CopyTradePlanController extends Controller
{
    public function __construct()
    {
        $this->middleware(['registeredUser', 'kycVerification']);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $user = Auth::user();

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('user.dashboard')],
            ['label' => 'Copy Trade Plans', 'active' => true]
        ];

        $copyTradePlans = CopyTradePlan::latest()->get();

        $data = [
            'title' => 'Copy Trade Plans',
            'user' => $user,
            'breadcrumbs' => $breadcrumbs,
            'copyTradePlans' => $copyTradePlans
        ];

        return view('dashboard.user.copy-trade-plan.index', $data);
    }

    /**
     * Display user subscriptions
     */
    public function subscriptions()
    {
        $user = Auth::user();

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('user.dashboard')],
            ['label' => 'Copy Trade Subscriptions', 'active' => true]
        ];

        $userCopyTradePlans = UserCopyTradePlan::where('user_id', $user->id)
            ->with('copyTradePlan')
            ->latest()
            ->get()
            ->map(function ($subscription) {

                // Safety checks
                if (!$subscription->ends_at) {
                    $subscription->progress_percentage = 0;
                    return $subscription;
                }

                $totalDays = $subscription->created_at->diffInDays($subscription->ends_at);
                $daysElapsed = $subscription->created_at->diffInDays(now());

                $subscription->progress_percentage = $totalDays > 0
                    ? min(100, ($daysElapsed / $totalDays) * 100)
                    : 0;

                return $subscription;
            });

        return view('dashboard.user.copy-trade-plan.subscriptions', [
            'title' => 'Copy Trade Subscriptions',
            'user' => $user,
            'breadcrumbs' => $breadcrumbs,
            'userCopyTradePlans' => $userCopyTradePlans
        ]);
    }


    /**
     * Store a new copy trade plan subscription
     */
    public function store(Request $request, string $uuid)
    {
        $copyTradePlan = CopyTradePlan::where('uuid', $uuid)->firstOrFail();

        // Build dynamic field name
        $field = 'amount_' . $uuid;

        $request->validate([
            $field => ['required', 'numeric', 'min:' . $copyTradePlan->minimum, 'max:' . $copyTradePlan->maximum],
        ], [
            $field . '.min' => 'Amount is below the minimum required for this plan.',
            $field . '.max' => 'Amount exceeds the maximum allowed for this plan.',
            $field . '.required' => 'Amount is required.',
            $field . '.numeric' => 'Amount must be a number.',
        ]);

        $amount = $request->input($field);
        $user = Auth::user();

        // Check if user already has an active copy trade plan
        $activePlan = UserCopyTradePlan::where('user_id', $user->id)
            ->where('status', UserCopyTradePlanStatus::ACTIVE)
            ->exists();

        if ($activePlan) {
            return redirect()->route('user.copy-trade-plan.index')
                ->with('error', 'You already have an active copy trade plan. Please cancel it before subscribing to a new one.');
        }

        // Check if user has enough wallet balance
        if ($user->wallet_balance < $amount) {
            return redirect()->route('user.copy-trade-plan.index')
                ->with('error', 'Insufficient wallet balance for this plan subscription.');
        }

        $meta['copyTradePlan'] = [
            'plan_name' => $copyTradePlan->name,
            'plan_type' => $copyTradePlan->type,
            'plan_min_amount' => $copyTradePlan->minimum,
            'plan_max_amount' => $copyTradePlan->maximum,
            'plan_daily_profit' => $copyTradePlan->daily_profit,
            'plan_commission_percentage' => $copyTradePlan->commission_percentage,
            'plan_duration_days' => $copyTradePlan->daily_profit_duration,
            'plan_trades_per_day' => $copyTradePlan->trades_per_day,
            'plan_features' => $copyTradePlan->features,
        ];

        try {
            DB::beginTransaction();

            $user = User::where('id', Auth::id())->lockForUpdate()->first();

            // Deduct from wallet balance (financial logic preserved)
            if ($user->wallet_balance < $amount) {
                 throw new \Exception('Insufficient wallet balance during processing.');
            }
            $user->wallet_balance -= $amount;
            $user->save();

            // Create user copy trade plan subscription
            $userCopyTradePlan = UserCopyTradePlan::create([
                'user_id' => $user->id,
                'copy_trade_plan_id' => $copyTradePlan->id,
                'amount' => $amount,
                'daily_profit' => $copyTradePlan->daily_profit,
                'commission_percentage' => $copyTradePlan->commission_percentage,
                'duration' => $copyTradePlan->daily_profit_duration, 
                'trades_per_day' => $copyTradePlan->trades_per_day ?? 1,
                'starts_at' => now(),
                'ends_at' => now()->addDays((int) $copyTradePlan->daily_profit_duration),
                'status' => UserCopyTradePlanStatus::ACTIVE
            ]);

            // Log transaction
            Transaction::create([
                'user_id' => $user->id,
                'type' => TransactionType::PLAN_SUBSCRIPTION,
                'direction' => TransactionDirection::DEBIT,
                'description' => 'Copy Trade Plan Subscription',
                'amount' => $amount,
                'method' => TransactionMethod::CRYPTO,
                'transaction_at' => now(),
                'reference_id' => generateReferenceId(),
                'other_reference_id' => $userCopyTradePlan->uuid,
                'meta' => $meta,
                'status' => TransactionStatus::COMPLETED
            ]);

            DB::commit();

            return redirect()->route('user.copy-trade-plan.index') 
                ->with('success', 'Copy Trade Plan subscribed successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->route('user.copy-trade-plan.index')
                ->with('error', 'Copy Trade Plan subscription failed');
        }
    }

    /**
     * Cancel a copy trade plan subscription
     */
    public function cancel(string $uuid)
    {
        $user = Auth::user();
        $userCopyTradePlan = UserCopyTradePlan::where('uuid', $uuid)
            ->where('user_id', $user->id)
            ->where('status', UserCopyTradePlanStatus::ACTIVE)
            ->firstOrFail();

        try {
            DB::beginTransaction();

             $user = User::where('id', Auth::id())->lockForUpdate()->first();

            // Refund logic (Capital refund to wallet_balance)
            $user->wallet_balance += $userCopyTradePlan->amount;
            
            $user->save();

            $userCopyTradePlan->status = UserCopyTradePlanStatus::CANCELLED;
            $userCopyTradePlan->save();

            Transaction::create([
                'user_id' => $user->id,
                'type' => TransactionType::PLAN_SUBSCRIPTION,
                'direction' => TransactionDirection::CREDIT,
                'description' => 'Copy Trade Plan Cancellation - Capital Refund',
                'amount' => $userCopyTradePlan->amount,
                'method' => TransactionMethod::CRYPTO,
                'transaction_at' => now(),
                'reference_id' => generateReferenceId(),
                'other_reference_id' => $userCopyTradePlan->uuid,
                'status' => TransactionStatus::COMPLETED
            ]);

            DB::commit();

            return redirect()->route('user.copy-trade-plan.subscriptions')
                ->with('success', 'Copy Trade Plan cancelled successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->back()->with('error', 'Cancellation failed');
        }
    }

    /**
     * Withdraw profit from a copy trade plan
     */
    public function withdrawProfit(string $uuid)
    {
        $user = Auth::user();
        $userCopyTradePlan = UserCopyTradePlan::where('uuid', $uuid)
            ->where('user_id', $user->id)
            ->where('status', UserCopyTradePlanStatus::ACTIVE)
            ->firstOrFail();

        if ($userCopyTradePlan->unrealized_profit <= 0) {
            return redirect()->back()->with('error', 'No unrealized profit available to withdraw.');
        }

        try {
            DB::beginTransaction();

            // Lock user for update
            $user = User::where('id', Auth::id())->lockForUpdate()->first();

            $amount = $userCopyTradePlan->unrealized_profit;

            // Debit from plan's unrealized profit (reset to 0)
            $userCopyTradePlan->unrealized_profit = 0;
            $userCopyTradePlan->save();

            // Credit to user wallet
            $user->wallet_balance += $amount;
            $user->save();

            // Log transaction
            Transaction::create([
                'user_id' => $user->id,
                'type' => TransactionType::PROFIT,
                'direction' => TransactionDirection::CREDIT,
                'description' => 'Copy Trade Profit Withdrawal - ' . $userCopyTradePlan->copyTradePlan->name,
                'amount' => $amount,
                'method' => TransactionMethod::CRYPTO,
                'transaction_at' => now(),
                'reference_id' => generateReferenceId(),
                'other_reference_id' => $userCopyTradePlan->uuid,
                'status' => TransactionStatus::COMPLETED
            ]);

            DB::commit();

            return redirect()->route('user.copy-trade-plan.subscriptions')
                ->with('success', 'Profit withdrawn to your wallet successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->back()->with('error', 'Withdrawal failed. Please try again.');
        }
    }
}
