<?php

namespace App\Http\Controllers\Dashboard\Admin;

use App\Enum\Currency;
use App\Models\Market;
use App\Models\TradeBot;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\Http\Requests\StoreTradeBotRequest;
use App\Http\Requests\UpdateTradeBotRequest;

class TradeBotController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Trade Bots', 'active' => true]
        ];

        $tradeBots = TradeBot::latest()->get();

        $market = new Market();

        $data = [
            'title' => 'Trade Bots',
            'breadcrumbs' => $breadcrumbs,
            'tradeBots' => $tradeBots,
            'market' => $market
        ];

        return view('dashboard.admin.trade_bot.index', $data);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Trade Bots', 'url' => route('admin.trade.bot.index')],
            ['label' => 'Create Trade Bot', 'active' => true]
        ];

        $currencies = Currency::cases();

        $markets = Market::all();

        $data = [
            'title' => 'Create Trade Bot',
            'breadcrumbs' => $breadcrumbs,
            'currencies' => $currencies,
            'markets' => $markets
        ];

        return view('dashboard.admin.trade_bot.create', $data);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreTradeBotRequest $request)
    {
        try {
            DB::transaction(function () use ($request) {

                $data = [
                    'name' => $request->name,
                    'minimum' => $request->minimum,
                    'maximum' => $request->maximum,
                    'currency' => $request->currency,
                    'asset_pairs' => $request->asset_pairs,
                    'features' =>  $request->features,
                    'parameters' => $request->parameters,
                    'description' => $request->description
                ];

                TradeBot::create($data);
            });

            return redirect()
                ->route('admin.trade.bot.index')
                ->with('success', 'Trade Bot created successfully');
        } catch (\Exception $e) {
            Log::error($e->getMessage());

            return redirect()
                ->back()
                ->with('error', 'Failed to create trade bot');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $uuid)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Trade Bots', 'url' => route('admin.trade.bot.index')],
            ['label' => 'Trade Bot Details', 'active' => true]
        ];

        $tradeBot = TradeBot::where('uuid', $uuid)->firstOrFail();

        $market = new Market();

        $data = [
            'title' => 'Trade Bot Details',
            'breadcrumbs' => $breadcrumbs,
            'tradeBot' => $tradeBot,
            'market' => $market
        ];

        return view('dashboard.admin.trade_bot.show', $data);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $uuid)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Trade Bots', 'url' => route('admin.trade.bot.index')],
            ['label' => 'Edit Trade Bot', 'active' => true]
        ];

        $currencies = Currency::cases();

        $tradeBot = TradeBot::where('uuid', $uuid)->firstOrFail();

        $markets = Market::all();

        $data = [
            'title' => 'Edit Trade Bot',
            'breadcrumbs' => $breadcrumbs,
            'currencies' => $currencies,
            'tradeBot' => $tradeBot,
            'markets' => $markets,
        ];

        return view('dashboard.admin.trade_bot.edit', $data);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateTradeBotRequest $request, string $uuid)
    {
        try {
            $tradeBot = TradeBot::where('uuid', $uuid)->firstOrFail();

            DB::transaction(function () use ($request, $tradeBot) {

                $data = [
                    'name' => $request->name,
                    'minimum' => $request->minimum,
                    'maximum' => $request->maximum,
                    'currency' => $request->currency,
                    'asset_pairs' => $request->asset_pairs,
                    'features' =>  $request->features,
                    'parameters' => $request->parameters,
                    'description' => $request->description
                ];

                $tradeBot->update($data);
            });

            return redirect()
                ->route('admin.trade.bot.index')
                ->with('success', 'Trade Bot updated successfully');
        } catch (\Exception $e) {
            Log::error($e->getMessage());

            return redirect()
                ->back()
                ->with('error', 'Failed to update trade bot');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function delete(string $uuid)
    {
        try {
            $tradeBot = TradeBot::where('uuid', $uuid)->firstOrFail();

            DB::transaction(function () use ($tradeBot) {
                $tradeBot->delete();
            });

            return redirect()
                ->route('admin.trade.bot.index')
                ->with('success', 'Trade Bot deleted successfully');
        } catch (\Exception $e) {
            Log::error($e->getMessage());

            return redirect()
                ->back()
                ->with('error', 'Failed to delete trade bot');
        }
    }
}
