<?php

namespace App\Http\Controllers\Dashboard\Admin;

use App\Models\User;
use Illuminate\Http\Request;
use App\Models\IsolatedMarginRequest;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\Enum\IsolatedMarginRequestStatus;
use App\Http\Requests\UpdateIsolatedMarginRequestRequest;

class IsolatedMarginRequestController extends Controller
{
    /**
     * Display a listing of isolated margin requests
     */
    public function index(Request $request)
    {
        $status = $request->status;

        $query = IsolatedMarginRequest::with(['user', 'wallet'])
            ->orderBy('created_at', 'desc');

        if ($status) {
            $query->where('status', $status);
        }

        $requests = $query->paginate(15);

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Admin', 'url' => route('admin.dashboard')],
            ['label' => 'Isolated Margin Requests', 'active' => true]
        ];

        $data = [
            'title' => 'Isolated Margin Requests',
            'requests' => $requests,
            'breadcrumbs' => $breadcrumbs,
        ];

        return view('dashboard.admin.isolated_margin_request.index', $data);
    }

    /**
     * Display the specified isolated margin request
     */
    public function show(string $uuid)
    {
        $isolatedMarginRequest = IsolatedMarginRequest::with(['user', 'wallet'])
            ->where('uuid', $uuid)
            ->firstOrFail();

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Admin', 'url' => route('admin.dashboard')],
            ['label' => 'Isolated Margin Requests', 'url' => route('admin.isolated_margin_request.index')],
            ['label' => 'Request Details', 'active' => true]
        ];

        $data = [
            'title' => 'Isolated Margin Request Details',
            'isolatedMarginRequest' => $isolatedMarginRequest,
            'breadcrumbs' => $breadcrumbs,
            'statuses' => IsolatedMarginRequestStatus::cases(),
        ];

        return view('dashboard.admin.isolated_margin_request.show', $data);
    }

    /**
     * Update the isolated margin request
     */
    public function update(UpdateIsolatedMarginRequestRequest $request, string $uuid)
    {
        try {
            $isolatedMarginRequest = IsolatedMarginRequest::where('uuid', $uuid)->firstOrFail();

            DB::transaction(function () use ($isolatedMarginRequest, $request) {
                $isolatedMarginRequest->update([
                    'status' => $request->status,
                    'admin_note' => $request->admin_note,
                    'reviewed_at' => now(),
                ]);
            });

            $statusLabel = IsolatedMarginRequestStatus::from($request->status)->label();

            return redirect()->route('admin.isolated_margin_request.index')
                ->with('success', "Request has been {$statusLabel} successfully.");
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return redirect()->route('admin.isolated_margin_request.show', $uuid)
                ->with('error', 'Unable to update request.');
        }
    }
}
