<?php

namespace App\Http\Controllers\Dashboard\Admin;

use App\Models\User;
use Illuminate\Http\Request;
use App\Models\CopyTradeRequest;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\Enum\CopyTradeRequestStatus;
use App\Http\Requests\UpdateCopyTradeRequestRequest;

class CopyTradeRequestController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $status = $request->status;

        $query = CopyTradeRequest::with(['user', 'trader.user'])
            ->orderBy('created_at', 'desc');

        if ($status) {
            $query->where('status', $status);
        }

        $requests = $query->paginate(15);

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Admin', 'url' => route('admin.dashboard')],
            ['label' => 'Copy Trade Requests', 'active' => true]
        ];

        $data = [
            'title' => 'Copy Trade Requests',
            'requests' => $requests,
            'breadcrumbs' => $breadcrumbs,
        ];

        return view('dashboard.admin.copy_trade_request.index', $data);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $uuid)
    {
        $copyTradeRequest = CopyTradeRequest::with(['user', 'trader.user'])
            ->where('uuid', $uuid)
            ->firstOrFail();

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Admin', 'url' => route('admin.dashboard')],
            ['label' => 'Copy Trade Requests', 'url' => route('admin.copy.trade.request.index')],
            ['label' => 'Request Details', 'active' => true]
        ];

        $data = [
            'title' => 'Copy Trade Request Details',
            'copyTradeRequest' => $copyTradeRequest,
            'breadcrumbs' => $breadcrumbs,
            'statuses' => CopyTradeRequestStatus::cases(),
        ];

        return view('dashboard.admin.copy_trade_request.show', $data);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateCopyTradeRequestRequest $request, string $uuid)
    {
        try {
            $copyTradeRequest = CopyTradeRequest::where('uuid', $uuid)->firstOrFail();

            DB::transaction(function () use ($copyTradeRequest, $request) {
                $copyTradeRequest->update([
                    'status' => $request->status,
                    'admin_note' => $request->admin_note,
                    'reviewed_at' => now(),
                ]);
            });

            $statusLabel = CopyTradeRequestStatus::from($request->status)->label();

            return redirect()->route('admin.copy.trade.request.index')
                ->with('success', "Request has been {$statusLabel} successfully.");
        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return redirect()->route('admin.copy.trade.request.show', $uuid)
                ->with('error', 'Unable to update request.');
        }
    }
}
