<?php

namespace App\Http\Controllers\Dashboard\Admin;

use App\Models\CopyTradePlan;
use App\Enum\PlanType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\Models\UserCopyTradePlan;
use App\Enum\TransactionType;
use App\Enum\TransactionDirection;
use App\Enum\TransactionMethod;
use App\Models\Transaction;
use App\Enum\TransactionStatus;
use App\Enum\UserCopyTradePlanStatus;
use App\Http\Requests\StoreCopyTradePlanRequest;
use App\Http\Requests\UpdateCopyTradePlanRequest;

class CopyTradePlanController extends Controller
{
    public function index()
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Copy Trade Plans', 'active' => true]
        ];

        $copyTradePlans = CopyTradePlan::latest()->get();

        $data = [
            'title' => 'Copy Trade Plans',
            'breadcrumbs' => $breadcrumbs,
            'copyTradePlans' => $copyTradePlans,
        ];

        return view('dashboard.admin.copy_trade_plan.index', $data);
    }

    public function create()
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Copy Trade Plans', 'url' => route('admin.copy.trade.plan.index')],
            ['label' => 'Create Plan', 'active' => true]
        ];

        $planTypes = PlanType::cases();

        $data = [
            'title' => 'Create Copy Trade Plan',
            'breadcrumbs' => $breadcrumbs,
            'planTypes' => $planTypes

        ];

        return view('dashboard.admin.copy_trade_plan.create', $data);
    }

    public function store(StoreCopyTradePlanRequest $request)
    {
        $validated = $request->validated();

        try {
            DB::beginTransaction();


            CopyTradePlan::create($validated);

            DB::commit();

            return redirect()->route('admin.copy.trade.plan.index')
                ->with('success', 'Copy Trade Plan created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage()); // Updated to getMessage() for cleaner logs
            return redirect()->back()->with('error', config('app.messages.error'));
        }
    }

    public function edit(CopyTradePlan $copyTradePlan)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Copy Trade Plans', 'url' => route('admin.copy.trade.plan.index')],
            ['label' => 'Edit Plan', 'active' => true]
        ];

        $planTypes = PlanType::cases();

        $data = [
            'title' => 'Edit Copy Trade Plan',
            'breadcrumbs' => $breadcrumbs,
            'copyTradePlan' => $copyTradePlan,
            'planTypes' => $planTypes
        ];

        return view('dashboard.admin.copy_trade_plan.edit', $data);
    }

    public function update(UpdateCopyTradePlanRequest $request, CopyTradePlan $copyTradePlan)
    {
        $validated = $request->validated();

        try {
            DB::beginTransaction();

            $copyTradePlan->update($validated);

            DB::commit();

            return redirect()->route('admin.copy.trade.plan.index')
                ->with('success', 'Copy Trade Plan updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->back()->with('error', config('app.messages.error'));
        }
    }

    public function show(CopyTradePlan $copyTradePlan)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Copy Trade Plans', 'url' => route('admin.copy.trade.plan.index')],
            ['label' => 'Plan Details', 'active' => true]
        ];

        // Load subscribers
        $subscribers = $copyTradePlan->userCopyTradePlans()
            ->with('user')
            ->latest()
            ->paginate(15);

        $data = [
            'title' => 'Copy Trade Plan Details',
            'breadcrumbs' => $breadcrumbs,
            'copyTradePlan' => $copyTradePlan,
            'subscribers' => $subscribers,
        ];

        return view('dashboard.admin.copy_trade_plan.show', $data);
    }

    public function terminate($uuid)
    {
        try {
            DB::beginTransaction();

            $subscription = UserCopyTradePlan::where('uuid', $uuid)->firstOrFail();

            if ($subscription->status !== UserCopyTradePlanStatus::ACTIVE) {
                return redirect()->back()->with('error', 'Only active subscriptions can be terminated.');
            }

            // Refund remaining amount logic (or full amount as per user request/standard)
            // Here assuming full refund of initial amount + return to wallet
            $user = $subscription->user;
            $user->wallet_balance += $subscription->amount;
            $user->save();

            $subscription->status = UserCopyTradePlanStatus::CANCELLED;
            $subscription->save();

             // Log cancellation transaction
             Transaction::create([
                'user_id' => $user->id,
                'type' => TransactionType::PLAN_SUBSCRIPTION,
                'direction' => TransactionDirection::CREDIT,
                'description' => 'Copy Trade Plan Terminated by Admin - Refund',
                'amount' => $subscription->amount,
                'method' => TransactionMethod::CRYPTO,
                'transaction_at' => now(),
                'reference_id' => generateReferenceId(),
                'other_reference_id' => $subscription->uuid,
                'status' => TransactionStatus::COMPLETED
            ]);

            DB::commit();

            return redirect()->back()->with('success', 'Subscription terminated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e);
            return redirect()->back()->with('error', 'Failed to terminate subscription.');
        }
    }

    public function addProfit(Request $request, $uuid)
    {
        $request->validate([
            'amount' => 'required|numeric',
        ]);

        try {
            DB::beginTransaction();

            $subscription = UserCopyTradePlan::where('uuid', $uuid)->firstOrFail();

            $subscription->unrealized_profit += $request->amount;
            $subscription->save();


             Transaction::create([
                'user_id' => $subscription->user_id,
                'type' => TransactionType::PROFIT,
                'direction' => $request->amount >= 0 ? TransactionDirection::CREDIT : TransactionDirection::DEBIT,
                'description' => 'Profit/Loss - ' . $subscription->copyTradePlan->name,
                'amount' => abs($request->amount),
                'method' => TransactionMethod::TRADE,
                'transaction_at' => now(),
                'reference_id' => generateReferenceId(),
                'other_reference_id' => $subscription->uuid,
                'status' => TransactionStatus::COMPLETED
            ]);

            DB::commit();

            return redirect()->back()->with('success', 'Profit added successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e);
            return redirect()->back()->with('error', 'Failed to add profit.');
        }
    }

    public function reactivate($uuid)
    {
        try {
            DB::beginTransaction();

            $subscription = UserCopyTradePlan::where('uuid', $uuid)->firstOrFail();

            if ($subscription->status !== UserCopyTradePlanStatus::CANCELLED) {
                return redirect()->back()->with('error', 'Only cancelled subscriptions can be reactivated.');
            }

            $user = $subscription->user;

            // Check if user has sufficient balance
            // if ($user->wallet_balance < $subscription->amount) {
            //     return redirect()->back()->with('error', 'User does not have sufficient balance (' . currency($user->currency) . formatAmount($subscription->amount) . ') to reactivate this plan.');
            // }

            // Deduct amount from wallet
            // $user->decrement('wallet_balance', $subscription->amount);

            // Update subscription status
            $subscription->status = UserCopyTradePlanStatus::ACTIVE;
            $subscription->save();

             // Log reactivation transaction
            //  Transaction::create([
            //     'user_id' => $user->id,
            //     'type' => TransactionType::PLAN_SUBSCRIPTION,
            //     'direction' => TransactionDirection::DEBIT,
            //     'description' => 'Copy Trade Plan Reactivated by Admin',
            //     'amount' => $subscription->amount,
            //     'method' => TransactionMethod::CRYPTO,
            //     'transaction_at' => now(),
            //     'reference_id' => generateReferenceId(),
            //     'other_reference_id' => $subscription->uuid,
            //     'status' => TransactionStatus::COMPLETED
            // ]);

            DB::commit();

            return redirect()->back()->with('success', 'Subscription reactivated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e);
            return redirect()->back()->with('error', 'Failed to reactivate subscription.');
        }
    }

    public function delete(CopyTradePlan $copyTradePlan)
    {
        try {
            DB::beginTransaction();

            $copyTradePlan->delete();

            DB::commit();

            return redirect()->route('admin.copy.trade.plan.index')
                ->with('success', 'Copy Trade Plan deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e);
            return redirect()->back()->with('error', config('app.messages.error'));
        }
    }
}
