<?php

namespace App\Http\Controllers\Dashboard\Admin;

use App\Models\User;
use App\Models\CopyTrade;
use Illuminate\Http\Request;
use App\Models\CopyTradeParticipant;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\Enum\CopyTradeStatus;
use App\Enum\CopyTradeRequestStatus;
use App\Enum\TransactionDirection;
use App\Enum\TransactionMethod;
use App\Enum\TransactionStatus;
use App\Enum\TransactionType;
use App\Http\Requests\StoreCopyTradeRequest;
use App\Models\Trader;
use App\Models\Transaction;

class CopyTradeController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $status = $request->status;

        $query = CopyTrade::withCount('participants')
            ->orderBy('created_at', 'desc');

        if ($status) {
            $query->where('status', $status);
        }

        $trades = $query->paginate(15);

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Admin', 'url' => route('admin.dashboard')],
            ['label' => 'Copy Trades', 'active' => true]
        ];

        $data = [
            'title' => 'Copy Trades',
            'trades' => $trades,
            'breadcrumbs' => $breadcrumbs,
        ];

        return view('dashboard.admin.copy_trade.index', $data);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $traders = Trader::with('user')->get();

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Admin', 'url' => route('admin.dashboard')],
            ['label' => 'Copy Trades', 'url' => route('admin.copy.trade.index')],
            ['label' => 'Create Trade', 'active' => true]
        ];

        $data = [
            'title' => 'Create Copy Trade',
            'breadcrumbs' => $breadcrumbs,
            'traders' => $traders,
        ];

        return view('dashboard.admin.copy_trade.create', $data);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreCopyTradeRequest $request)
    {
        try {
            DB::transaction(function () use ($request) {
                CopyTrade::create([
                    'trader_id' => $request->trader_id,
                    'name' => $request->name,
                    'description' => $request->description,
                    'initial_amount' => $request->initial_amount,
                    'duration_hours' => $request->duration_hours,
                    'status' => CopyTradeStatus::ACTIVE->value,
                    'started_at' => now(),
                ]);
            });

            return redirect()->route('admin.copy.trade.index')
                ->with('success', 'Copy trade created successfully.');
        } catch (\Exception $e) {
            Log::error($e);
            return redirect()->route('admin.copy.trade.create')
                ->with('error', 'Unable to create copy trade.');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $uuid)
    {
        $trade = CopyTrade::with(['participants.user', 'trader.user'])
            ->where('uuid', $uuid)
            ->firstOrFail();

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Admin', 'url' => route('admin.dashboard')],
            ['label' => 'Copy Trades', 'url' => route('admin.copy.trade.index')],
            ['label' => $trade->name, 'active' => true]
        ];

        $data = [
            'title' => $trade->name,
            'trade' => $trade,
            'breadcrumbs' => $breadcrumbs,
        ];

        return view('dashboard.admin.copy_trade.show', $data);
    }

    /**
     * Show participants selection page
     */
    public function participants(string $uuid)
    {
        $trade = CopyTrade::with(['participants.user'])
            ->where('uuid', $uuid)
            ->firstOrFail();

        // Get users with approved copy trade requests for this trader and connected wallets
        $eligibleUsers = User::whereHas('copyTradeRequests', function ($query) use ($trade) {
            $query->where('status', CopyTradeRequestStatus::APPROVED->value)
                ->where('trader_id', $trade->trader_id);
        })
            ->whereNotNull('wallet_address')
            ->whereDoesntHave('copyTradeParticipations', function ($query) use ($trade) {
                $query->where('copy_trade_id', $trade->id);
            })
            ->get();

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Admin', 'url' => route('admin.dashboard')],
            ['label' => 'Copy Trades', 'url' => route('admin.copy.trade.index')],
            ['label' => $trade->name, 'url' => route('admin.copy.trade.show', $trade->uuid)],
            ['label' => 'Manage Participants', 'active' => true]
        ];

        $data = [
            'title' => 'Manage Participants - ' . $trade->name,
            'trade' => $trade,
            'eligibleUsers' => $eligibleUsers,
            'breadcrumbs' => $breadcrumbs,
        ];

        return view('dashboard.admin.copy_trade.participants', $data);
    }

    /**
     * Add participants to trade
     */
    public function addParticipants(Request $request, string $uuid)
    {
        try {
            $trade = CopyTrade::where('uuid', $uuid)->firstOrFail();

            $request->validate([
                'user_ids' => 'required|array',
                'user_ids.*' => 'exists:users,id',
                'deduct_balance' => 'sometimes|boolean',
            ]);

            $deductBalance = $request->boolean('deduct_balance');

            DB::transaction(function () use ($trade, $request, $deductBalance, $uuid) {
                foreach ($request->user_ids as $userId) {
                    $user = User::findOrFail($userId);

                    // Check balance if deduction is requested
                    if ($deductBalance && $user->wallet_balance < $trade->initial_amount) {
                        Log::warning("User {$user->id} has insufficient balance for copy trade deduction");
                        return redirect()->route('admin.copy.trade.participants', $uuid)
                            ->with('error', 'User {$user->id} has insufficient balance for copy trade deduction');
                    }

                    CopyTradeParticipant::create([
                        'copy_trade_id' => $trade->id,
                        'user_id' => $userId,
                        'invested_amount' => $trade->initial_amount,
                        'is_active' => true,
                    ]);

                    // Deduct balance
                    if ($deductBalance) {
                        $user->decrement('wallet_balance', $trade->initial_amount);

                        // Record transaction
                        Transaction::create([
                            'user_id' => $user->id,
                            'amount' => $trade->initial_amount,
                            'description' => "Investment in copy trade: {$trade->name}",
                            'reference_id' => generateReferenceId(),
                            'type' => TransactionType::COPY_TRADE,
                            'direction' => TransactionDirection::DEBIT,
                            'status' => TransactionStatus::COMPLETED,
                            'method' => TransactionMethod::TRADE
                        ]);
                    }
                }
            });

            return redirect()->route('admin.copy.trade.show', $trade->uuid)
                ->with('success', 'Participants added successfully.');
        } catch (\Exception $e) {
            Log::error($e);
            return redirect()->route('admin.copy.trade.participants', $uuid)
                ->with('error', 'Unable to add participants. ' . $e);
        }
    }

    /**
     * Show profit/loss management page
     */
    public function profit(string $uuid)
    {
        $trade = CopyTrade::with(['participants.user'])
            ->where('uuid', $uuid)
            ->firstOrFail();

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Admin', 'url' => route('admin.dashboard')],
            ['label' => 'Copy Trades', 'url' => route('admin.copy.trade.index')],
            ['label' => $trade->name, 'url' => route('admin.copy.trade.show', $trade->uuid)],
            ['label' => 'Add Profit/Loss', 'active' => true]
        ];

        $data = [
            'title' => 'Add Profit/Loss - ' . $trade->name,
            'trade' => $trade,
            'breadcrumbs' => $breadcrumbs,
        ];

        return view('dashboard.admin.copy_trade.profit', $data);
    }

    /**
     * Add profit/loss to participants
     */
    public function addProfit(Request $request, string $uuid)
    {
        try {
            $trade = CopyTrade::where('uuid', $uuid)->firstOrFail();

            $request->validate([
                'profit_loss' => 'required|numeric',
                'apply_to' => 'required|in:all,selected',
                'participant_ids' => 'required_if:apply_to,selected|array',
            ]);

            DB::transaction(function () use ($trade, $request) {
                $query = CopyTradeParticipant::where('copy_trade_id', $trade->id);

                if ($request->apply_to == 'selected') {
                    $query->whereIn('id', $request->participant_ids);
                }

                $participants = $query->get();

                foreach ($participants as $participant) {
                    // Update participant's profit/loss and unrealized profit
                    $participant->update([
                        'profit_loss' => $participant->profit_loss + $request->profit_loss,
                        'unrealized_profit' => $participant->unrealized_profit + $request->profit_loss,
                    ]);

                    $user = $participant->user;

                    // Record transaction
                    Transaction::create([
                        'user_id' => $user->id,
                        'reference_id' => generateReferenceId(),
                        'amount' => abs($request->profit_loss),
                        'description' => "Return from copy trade: {$trade->name}",
                        'type' => TransactionType::COPY_TRADE,
                        'direction' => $request->profit_loss >= 0 ? TransactionDirection::CREDIT : TransactionDirection::DEBIT,
                        'status' => TransactionStatus::COMPLETED,
                        'method' => TransactionMethod::TRADE
                    ]);
                }

                // Update trade's total profit/loss
                $trade->update([
                    'profit_loss' => $trade->profit_loss + ($request->profit_loss * $participants->count()),
                ]);
            });

            return redirect()->route('admin.copy.trade.show', $trade->uuid)
                ->with('success', 'Profit/Loss added successfully to participants.');
        } catch (\Exception $e) {
            Log::error($e);
            return redirect()->route('admin.copy.trade.profit', $uuid)
                ->with('error', 'Unable to add profit/loss.');
        }
    }

    /**
     * Complete a trade
     */
    public function complete(string $uuid)
    {
        try {
            $trade = CopyTrade::where('uuid', $uuid)->firstOrFail();

            DB::transaction(function () use ($trade) {
                $trade->update([
                    'status' => CopyTradeStatus::COMPLETED->value,
                    'ended_at' => now(),
                ]);

                // Mark all participants as inactive
                $trade->participants()->update(['is_active' => false]);
            });

            return redirect()->route('admin.copy.trade.show', $trade->uuid)
                ->with('success', 'Trade completed successfully.');
        } catch (\Exception $e) {
            Log::error($e);
            return redirect()->route('admin.copy.trade.show', $uuid)
                ->with('error', 'Unable to complete trade.');
        }
    }
}
