<?php

namespace App\Console\Commands;

use Carbon\Carbon;
use App\Models\UserCopyTradePlan;
use App\Models\Transaction;
use App\Models\CopyTradePlan;
use App\Models\User;
use App\Enum\UserCopyTradePlanStatus;
use App\Enum\TransactionType;
use App\Enum\TransactionMethod;
use App\Enum\TransactionStatus;
use Illuminate\Console\Command;
use App\Enum\TransactionDirection;
use Illuminate\Support\Facades\DB;

class ProcessCopyTradeProfit extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:process-copy-trade-profit';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Process profits and commissions for active copy trade plan subscriptions';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $today = Carbon::today();

        $userCopyTradePlans = UserCopyTradePlan::where('status', UserCopyTradePlanStatus::ACTIVE)
            ->whereDate('starts_at', '<=', $today)
            ->get();

        foreach ($userCopyTradePlans as $userCopyTradePlan) {
            // 1. Stop if plan has already ended
            if ($today->greaterThan($userCopyTradePlan->ends_at)) {
                $this->completePlan($userCopyTradePlan);
                continue;
            }

            // 2. Prevent double profit for same day
            $alreadyCredited = Transaction::where('user_id', $userCopyTradePlan->user_id)
                ->where('type', TransactionType::PROFIT)
                ->whereDate('created_at', $today)
                ->where('description', 'LIKE', '%Copy Trade Profit%')
                ->where('other_reference_id', $userCopyTradePlan->uuid)
                ->exists();

            if ($alreadyCredited) {
                continue;
            }

            // 3. Calculate daily profit
            (float) $profit = ($userCopyTradePlan->amount * $userCopyTradePlan->daily_profit) / 100;

            // Apply multiplication if set in plan (from new schema)
             if ($userCopyTradePlan->copyTradePlan->daily_profit_multiplication > 0) {
                 $profit = $profit * $userCopyTradePlan->copyTradePlan->daily_profit_multiplication;
             }
             
            
            $this->processCommissionForProfit($userCopyTradePlan, $profit);
            
            $userCopyTradePlan->increment('days_processed');

            // 4. If today is the final day, complete the plan after processing profit
            if ($today->greaterThanOrEqualTo($userCopyTradePlan->ends_at)) {
                $this->completePlan($userCopyTradePlan);
            }
        }

        return Command::SUCCESS;
    }

    protected function completePlan(UserCopyTradePlan $userCopyTradePlan)
    {
        DB::transaction(function () use ($userCopyTradePlan) {

            $user = User::find($userCopyTradePlan->user_id);

            // Refund the initial subscription amount to wallet balance
            $user->wallet_balance += $userCopyTradePlan->amount;
            $user->save();

            // Update plan status
            $userCopyTradePlan->status = UserCopyTradePlanStatus::COMPLETED;
            $userCopyTradePlan->save();

            // Log transaction
            Transaction::create([
                'user_id' => $userCopyTradePlan->user_id,
                'type' => TransactionType::PLAN_SUBSCRIPTION,
                'direction' => TransactionDirection::CREDIT,
                'description' => 'Copy Trade Plan Completed - Capital Refund',
                'amount' => $userCopyTradePlan->amount,
                'method' => TransactionMethod::CRYPTO,
                'transaction_at' => now(),
                'reference_id' => generateReferenceId(),
                'other_reference_id' => $userCopyTradePlan->uuid,
                'status' => TransactionStatus::COMPLETED
            ]);

            $this->info("Copy Trade Plan #{$userCopyTradePlan->uuid} completed and refunded.");
        });
    }

    /**
     * Process commission when profits are reported
     * This method should be called when a trader reports profits for today
     */
    public function processCommissionForProfit(UserCopyTradePlan $userCopyTradePlan, float $dailyProfit)
    {
        DB::transaction(function () use ($userCopyTradePlan, $dailyProfit) {

            if ($dailyProfit <= 0) {
                return; // No profit, no commission
            }

            $user = User::find($userCopyTradePlan->user_id);
            
            // Calculate commission based on plan's commission percentage
            $commission = ($dailyProfit * $userCopyTradePlan->commission_percentage) / 100;
            
            // Deduct commission from user's balance (goes to platform)
            $user->wallet_balance -= $commission;
            $user->save();

            // Add profit to unrealized gains on the plan
            $userCopyTradePlan->unrealized_profit += $dailyProfit;
            $userCopyTradePlan->save();

            $copyTradePlan = CopyTradePlan::find($userCopyTradePlan->copy_trade_plan_id);
            // Log commission transaction
            Transaction::create([
                'user_id' => $userCopyTradePlan->user_id,
                'type' => TransactionType::COMMISSION,
                'direction' => TransactionDirection::DEBIT,
                'description' => 'Copy Trade Commission - ' . $copyTradePlan->name,
                'amount' => $commission,
                'method' => TransactionMethod::TRADE,
                'transaction_at' => now(),
                'reference_id' => generateReferenceId(),
                'other_reference_id' => $userCopyTradePlan->uuid,
                'status' => TransactionStatus::COMPLETED
            ]);

            $copyTradePlan = CopyTradePlan::find($userCopyTradePlan->copy_trade_plan_id);
            // Log profit transaction
            Transaction::create([
                'user_id' => $userCopyTradePlan->user_id,
                'type' => TransactionType::PROFIT,
                'direction' => TransactionDirection::CREDIT,
                'description' => 'Copy Trade Profit - ' . $copyTradePlan->name,
                'amount' => $dailyProfit,
                'method' => TransactionMethod::TRADE,
                'transaction_at' => now(),
                'reference_id' => generateReferenceId(),
                'other_reference_id' => $userCopyTradePlan->uuid,
                'status' => TransactionStatus::COMPLETED
            ]);
        });
    }
}
